import binascii
import time
from typing import Any,List
from cacheout import LFUCache
from collections import namedtuple
from struct import Struct
from enum import Enum
from lib.req import ReqVehicleInfo,ReqBlackList,ReqMac2
from lib.frame import TerFrame
from lib.struct import Mac2,SerialNumber
from lib.format import FORMAT_PAY_DATA
from lib.push import PushQueueOperate,PushOperateManager,PushDevOnline
from utils import OnPlateRemoveRedundant,FORMAT_TIME
from set import ENABLE_PLATES_DIFF,PARK_ID,DEV_NUMBER,RSU_ID,PARK_ID
from log import Logger
logger = Logger(__name__)


class HandlerErrCode(Enum):
    NOT_SELECT_OBU = 0 # 没有找到OBU
    OK = 1 #正常处理
    PAY_OK = 2 # 正常扣费
    RUN_ERR = -1 # 处理时捕获异常
    STATUS_ERR = -2 # 状态异常
    PLATE_NOT_MATCH = -3 # 车牌不符
    NET_WORK_ERR = -4 #网络请求异常
    PUSH_ERR = -5 #入队异常
    UNKNOWN_ERR = -6 # 未知错误
    EXCESS_ERR = -7 # 超额消费



ISSUED_MAP = {
            "北京": "11", "天津": "12", "河北": "13", "山西": "14", "内蒙": "15",
            "辽宁": "21", "吉林": "22", "龙": "23","龙江":"23", "上海": "31", "江苏": "32",
            "浙江": "33", "安徽": "34", "福建": "35", "江西": "36", "山东": "37", "河南": "41",
            "湖北": "42", "湖南": "43", "广东": "44", "广西": "45", "海南": "46",
            "重庆": "50", "贵州": "52", "云南": "53", "西藏": "54",
            "陕西": "61", "甘肃": "62", "青海": "63", "宁夏": "64", "新疆": "65",
            "台湾地区": "71", "香港": "81", "澳门": "82"}
        

def BIT_TO_NAMEEDTUPLE(bit:bytes,field:List[str],fmt:str)->Any:
    '''
    @bit:要解析的byte字串
    @field:格式化的字段
    @fmt:格式化规则
    返回一个命名元祖对象
    '''
    nt = namedtuple("o",field)
    param = nt(*Struct(fmt).unpack(bit))
    return param




async def HANDLER_FIND_OBU_SIGNAL(stream, cache,handlerCache, rf):
    '''搜索OBU信号'''
    logger.info(f"--------IN E4 TIME:{time.time()}")
    data = rf.data
    status, = Struct("!s").unpack(data[0:1])
    if status.hex() != "00":
        return HandlerErrCode.NOT_SELECT_OBU.value
    
    # 判定超额消费
    order = cache.get("order")
    if order["fee"] >= 5000:
        return HandlerErrCode.EXCESS_ERR.value
    
    param = BIT_TO_NAMEEDTUPLE(data[1:],["obuMac","zw","obuInfo","zw1","cardInfo","zw3","vehicleInfo"],"!4ss27ss43ss96s")
    cache.set("obuId", param.obuMac.hex())
    workKey = cache.get("workKey")
    # 解析系统信息
    sysParam = BIT_TO_NAMEEDTUPLE(param.obuInfo,[
        "serviceCode",
        "agrType",
        "agrVersion",
        "agrNumber",
        "startDate",
        "expiredDate",
        "isDisassembly"],
        "!8sss8s4s4ss")

    vehicleParam = BIT_TO_NAMEEDTUPLE(param.vehicleInfo,["ciphertext","random","zw"],"!80s8s8s")
    
    
    futureVehicleRes = ReqVehicleInfo(
        RSU_ID,
        vehicleParam.ciphertext.hex(),
        vehicleParam.random.hex(),
        sysParam.agrNumber.hex(),
        sysParam.serviceCode.hex()[0:8],
        workKey.key
    )

    cardParam = BIT_TO_NAMEEDTUPLE(param.cardInfo,[
        "issued",
        "cardType",
        "cardVersion",
        "cardNetworkNumber",
        "cardNumber",
        "startTime",
        "endTime",
        "plate",
        "fic"
        ],"!8sss2s8s4s4s12s3s")

    issuedKey = binascii.unhexlify(cardParam.issued.hex()[slice(0, 8)]).decode('gb2312')  #修正卡发行标识
    issued = ISSUED_MAP.get(issuedKey)
    if not issued:
        logger.error(f"读取到不明省份车辆:{issuedKey}")
        return HandlerErrCode.RUN_ERR.value
    try:
        cardParam = cardParam._replace(cardNetworkNumber=Struct("!2s").pack(bytes.fromhex(issued + cardParam.cardNetworkNumber.hex()[2:])))
    except Exception as e:
        logger.error(f"卡网络编号修正异常:{e}")
        return HandlerErrCode.RUN_ERR.value

    # 异步卡信息初始化完成就发出黑名单请求
    futureBalckListRes = ReqBlackList(
        PARK_ID,
        cardParam.cardNumber.hex(),
        sysParam.agrNumber.hex(),
        cardParam.cardNetworkNumber.hex()
    )
    try:
        vehicleRes = await futureVehicleRes
    except Exception as e:
        logger.error(f"车辆解密异常:{e}")
        return HandlerErrCode.NET_WORK_ERR.value

    if vehicleRes == -1:
        logger.error("车辆解密失败")
        return HandlerErrCode.NET_WORK_ERR.value
    try:
        balckListRes = await futureBalckListRes
    except Exception as e:
        logger.error(f"黑名单请求异常:{e}")
        return HandlerErrCode.NET_WORK_ERR.value

    if balckListRes == -1:
        logger.error("黑名单请求失败")
        return HandlerErrCode.NET_WORK_ERR.value

    if balckListRes != "0":
        logger.warning(f"车辆黑名单内:{balckListRes}")
        return HandlerErrCode.NET_WORK_ERR.value
    try:
        vehicleInfo = BIT_TO_NAMEEDTUPLE(bytes.fromhex(vehicleRes['vehInfo'][0:118]), #只要前118个字符59个字节
        [
            "plate", # 车牌号
            "color", # 车牌颜色
            "type",# 车型
            "userType",# 车辆用户类型
            "size", #车辆尺寸(长2字节宽1字节高1字节单位:dm)
            "tiresNumber", # 车轮数
            "axleNumber", # 车轴数
            "axleWidth", #轴距,单位:dm
            "load", # 车辆载重/座位数，其中,载重的单位为:kg
            "feature", # 车辆特征描述
            "engineNumber" # 车辆发动机号
        ],"!12s2sss4sss2s3s16s16s")
    except Exception as e:
        logger.error(f"格式化参数异常:{e}")
        return HandlerErrCode.RUN_ERR.value

    try:
        # print(f"车牌:{vehicleInfo.plate}")
        vehiclePlate = OnPlateRemoveRedundant(vehicleInfo.plate.hex())
        vehicleInfo = vehicleInfo._replace(
                plate=Struct("!12s").pack(vehiclePlate.encode('gbk'))) #除杂后修正车牌号
    except Exception as e:
        logger.error(f"车牌除杂修正异常:{e}")
        return HandlerErrCode.RUN_ERR.value
    if ENABLE_PLATES_DIFF == 1:
        if vehiclePlate != order["plate"]:
            logger.info(f"读取车牌:{vehiclePlate}")
            return HandlerErrCode.PLATE_NOT_MATCH.value
    workKey = cache.get("workKey")
    tf = TerFrame(b'\x51',(order['fee'],bytes.fromhex(workKey.termId)),"!i6s")
    tf.rsctl = rf.rsctl
    handlerCache.set_many(dict(sysParam=sysParam, vehicleInfo=vehicleInfo, cardParam=cardParam))
    stream.write(tf.frame)
    return HandlerErrCode.OK.value


async def HANDLER_GET_MAC(stream, cache,handlerCache, rf):
    '''获取mac2'''
    logger.info(f"--------IN E7 TIME:{time.time()}")
    data = rf.data
    status, = Struct("!s").unpack(data[0:1])
    if status.hex() != "00":
        return HandlerErrCode.STATUS_ERR.value


    ORDER = cache.get("order")
    workKey = cache.get("workKey")
    cardParam = handlerCache.get("cardParam")
    # |①OBU物理地址(mac)|②消费前余额|③伪随机数|⑤卡片交易序号|
    param = BIT_TO_NAMEEDTUPLE(data[1:len(data)],["obuMac","banlance","passtypeRand","cardSN"],"!4s4s4s2s")
    transacDate = FORMAT_TIME("%Y%m%d")
    transacTime = FORMAT_TIME("%H%M%S")
    transactionTime = f"{transacDate}{transacTime}"
    feeHex = Struct("!i").pack(ORDER['fee']).hex()
    futureMacRes =  ReqMac2(RSU_ID,transactionTime,param.passtypeRand,param.cardSN,feeHex,cardParam,workKey.key)

    handlerCache.set("transactionTime", transactionTime)
    handlerCache.set("transacTime",transacTime)
    handlerCache.set("cardSN", param.cardSN)
    handlerCache.set("passtypeRand", param.passtypeRand)
    handlerCache.set("preBalance",param.banlance)

    macRes = await futureMacRes
    if macRes == -1:
        logger.error("mac请求异常")
        return HandlerErrCode.NET_WORK_ERR.value
    mac = Mac2(macRes["mac"])
    handlerCache.set("termNo", mac.termNo)
    tf = TerFrame(
        b'\x52',
        (
            bytes.fromhex(mac.termNo),
            bytes.fromhex(transactionTime),
            bytes.fromhex(mac.mac2)
        ),"!4s7s4s")
    tf.rsctl = rf.rsctl
    stream.write(tf.frame)
    return HandlerErrCode.OK.value

async def HANDLER_OVER_PAY(stream, cache,handlerCache, rf):
    '''完成消费电子记账'''
    logger.info(f"--------IN E5 TIME:{time.time()}")
    data = rf.data
    status, = Struct("!s").unpack(data[0:1])
    if status.hex() != "00":
        return HandlerErrCode.STATUS_ERR.value
    param = BIT_TO_NAMEEDTUPLE(data[1:len(data)],["obuMac","mac2","tac","banlance"],"!4s4s4s4s")
    transac = SerialNumber()
    transacNo = transac.create(PARK_ID)
    handlerCache.set("transacNo",transacNo)
    ORDER = cache.get("order")    
    workKey = cache.get("workKey")
    transacBody = FORMAT_PAY_DATA(
        PARK_ID,
        ORDER,
        workKey,
        transacNo,
        transac.randint, param.obuMac.hex(),
        param.banlance.hex(), param.tac.hex(), handlerCache) #原生交易数据
    enqueueStatus = 1
    pushQueueOperate=PushQueueOperate()
    pushOperateManager = PushOperateManager(pushQueueOperate=pushQueueOperate)
    await pushOperateManager.PushQueue(ORDER['payNo'],transacNo,transacBody)  # 即时向平台提交
    if pushOperateManager.pushQueueOperate.pushStatus == 1: #补传成功
         enqueueStatus = 2
    elif pushOperateManager.pushQueueOperate.pushStatus == 2: #补传依然失败
        return HandlerErrCode.PUSH_ERR.value
    handlerCache.set("enqueueStatus",enqueueStatus)

    return HandlerErrCode.PAY_OK.value
    

async def HANDLER_HEARTBEAT(stream, cache,handlerCache, rf):
    '''天线心跳'''
    # 增加状态上报机制 每十个心跳单位一次
    devStatus = cache.get("devStatus")
    if not devStatus:
        res = await PushDevOnline(DEV_NUMBER)
        cache.set("devStatus", "ok", ttl=60)
    return HandlerErrCode.OK.value

async def HANDLER_SET_NUM(stream, cache,handlerCache,rf):
    '''设置功率'''
    status, = Struct("!s").unpack(rf.data)
    if status.hex() != "00":
        logger.error(f"设置功率时状态异常:{status.hex()}")
        return HandlerErrCode.STATUS_ERR.value

    logger.info("设置功率完成...")
    tf = TerFrame(b'\x19',(1,),"!b")
    stream.write(tf.frame)
    return HandlerErrCode.OK.value

async def HANDLER_SET_CHANNEL(stream, cache,handlerCache,rf):
    '''设置信道'''
    status, = Struct("!s").unpack(rf.data)
    if status.hex() != "00":
        logger.error(f"设置功率时状态异常:{status.hex()}")
        return HandlerErrCode.STATUS_ERR.value
    logger.info("设置信道完成...")
    return HandlerErrCode.OK.value




HADLER_MAP = {
    "98": HANDLER_SET_NUM,# 设置档位
    "99": HANDLER_SET_CHANNEL,# 设置信道
    "d0": HANDLER_FIND_OBU_SIGNAL,  # 继续交易
    "d1": HANDLER_GET_MAC,  # 计算MAC2
    "d2": HANDLER_OVER_PAY,  # 消费
    "91": HANDLER_HEARTBEAT  # 心跳
}
