import json
import time
from typing import Any
from enum import Enum
from lib.proxy import ProxyDataPlatform,OAuthModeEnum
from lib.alert import AlarmMsgManager,AlarmLevel,AlarmType,AlarmContentTemplate
from set import RECORD_MODULE_PROTOCOL,RECORD_MODULE_HOST,RECORD_MODULE_PORT,PARK_ID,ADDRESSEE,DEV_NUMBER
from utils import FORMAT_TIME
from log import Logger
logger = Logger(__name__)

class Api(Enum):
    PUSH_IN_QUEUE = "/api/v1/park/order/controller/enqueue/"
    PUSH_IN_LOCAL_FAIL_LISTA = "/api/v1/terminal/record/service/failed/transac/"
    PUSH_BOOK_KEEP = "/api/v1/terminal/record/service/keep/"
    PUSH_DEV_ONLANE = "/api/v1/dev/platform/park/dev/status/push/"

async def PushDevOnline(devNumber)->Any:
    '''
    通知平台设备上线
    '''
    body = {
        "push_time": int(time.time()),
        "dev_number": devNumber,
        "dev_type": "Terminal",
        "is_one": "1",
        "status_info": {}
    }
    api = Api.PUSH_DEV_ONLANE.value
    httpProxy = ProxyDataPlatform(proxyOAuth=OAuthModeEnum.V2.value)
    res = await httpProxy.post(api,body)
    if httpProxy.isErr:
        logger.error(httpProxy.errMsg)
        return -1
    return res


class PushQueueOperate(object):
    def __init__(self,pushUrl="") -> None:
        self.pushUrl = pushUrl
        self.pushStatus = 0 #0;云端推送成功,1;本地推送成功,2;失败
        self.failAlarmManger = None

    
    def SandPushFailAlarm(self):
        if not self.failAlarmManger:
            return
        self.failAlarmManger.send(ADDRESSEE)

    async def PushError(self,payNo:str,transacNo:str,transacBody:dict):
        '''交易原始数据提交失败,推送进本地失败列表等待二次提交'''
        api = Api.PUSH_IN_LOCAL_FAIL_LISTA.value
        _t = int(time.time())
        body = {
                "item":dict(payNo=payNo,
                transNumber=transacNo,
                insertDate=FORMAT_TIME("%Y-%m-%d"),
                insertTime=_t,
                insertHour=time.localtime(_t).tm_hour,
                isUpload=0,
                isDelete=0,
                devNumber=DEV_NUMBER),
                "content":transacBody
            }

        httpProxy = ProxyDataPlatform()
        httpProxy.protocol = RECORD_MODULE_PROTOCOL  # 本地平台
        httpProxy.host = RECORD_MODULE_HOST
        httpProxy.port = RECORD_MODULE_PORT

        res = await httpProxy.post(api,body,timeOut=2)
        if httpProxy.isErr:
            logger.error(f"提交源数据入本地队列异常:{httpProxy.errMsg}")
            return -1
        return res

    async def push(self,payNo:str,transacNo:str,transacBody:dict):
        data = {
        "pay_no": payNo,
        "transac_no": transacNo,
        "json_str": bytes(json.dumps(transacBody), encoding='gbk').hex(),
        }

        body = {
            "trans_list":[data]
        }

        httpProxy = ProxyDataPlatform(OAuthModeEnum.V2.value)
        # httpProxy.protocol = 'http'
        # httpProxy.host = '39.104.106.82'# 临时需要
        # httpProxy.port = 9000

        res = await httpProxy.post(self.pushUrl,body,timeOut=2)
        if httpProxy.isErr:
            logger.error(f"原始交易数据推送云队列异常:{httpProxy.errMsg}")
            return -1
        return res



class PushBookKeepOperate(object):
    def __init__(self,pushUrl: str) -> None:
        self.pushUrl = pushUrl
        self.pushTime =  int(time.time())
        self.timeRelated = {
            "pushDate": FORMAT_TIME("%Y-%m-%d"),
            "pushTime": self.pushTime,
            "pushHour": time.localtime(self.pushTime).tm_hour
        }

    def FormatSummaryData(self,handlerCache,order):
        '''
        2022-09-26
        取消vehicle字段,改为plate
        取消payInfo,不再上传关于OBU内部信息
        '''
        transacNo = handlerCache.get("transacNo")
        enqueueStatus = handlerCache.get("enqueueStatus")
        summary = {
            "payNo":order['payNo'],
            "transNumber":transacNo,
            "fee":order['fee'],
            "plate":order['plate'],
            "serviceId":PARK_ID,
            "devNumber":DEV_NUMBER,
            "hour":self.timeRelated["pushHour"],
            "enqueueStatus":enqueueStatus,
            "shift":2,
            "uploadTime":self.pushTime,
            "uploadDate":self.timeRelated["pushDate"]
        }
        return summary
    
    def PushError(self,payNo: str):
        #记账失败发送告警
        title = f"【{AlarmLevel.SERIOUS.value}-{AlarmType.NOT_RECORD_MODULE_REPLY.value}】"
        alarmMsg = AlarmContentTemplate(PARK_ID,payNo,FORMAT_TIME("%Y-%m-%d %H:%M:%S"))
        alarmMsgManager = AlarmMsgManager(title=title,msg=alarmMsg)
        alarmMsgManager.send(ADDRESSEE)

    async def push(self,content: dict,requestTime:int,realRequestTime:int):
        body = {"item":dict(
            payNo=content["payNo"],
            transNumber=content["transNumber"],
            insertdate=self.timeRelated["pushDate"],
            insertTime=self.timeRelated["pushTime"],
            inserthour=self.timeRelated["pushHour"],
            isUpload=0,
            isDelete=0,
            serviceId = PARK_ID,
            requestTime = requestTime,
            realRequestTime = realRequestTime,
            devNumber=content["devNumber"]),
            "content":content}
        httpProxy = ProxyDataPlatform(OAuthModeEnum.V2.value)
        httpProxy.protocol = RECORD_MODULE_PROTOCOL  # 本地平台
        httpProxy.host = RECORD_MODULE_HOST
        httpProxy.port = RECORD_MODULE_PORT

        res = await httpProxy.post(self.pushUrl,body,timeOut=2)
        if httpProxy.isErr: # 推送失败进行设置好的错误回调
            logger.error(f"记账推送时异常:{httpProxy.errMsg}")
            return -1
        return res


class PushOperateManager(object):
    def __init__(self,pushBookKeepOperate:PushBookKeepOperate = None,pushQueueOperate:PushQueueOperate=None) -> None:
        self.pushBookKeepOperate = pushBookKeepOperate
        self.pushQueueOperate = pushQueueOperate

    async def PushBookKeep(self,handlerCache,order,requestTime:int,realRequestTime:int):
        summary = self.pushBookKeepOperate.FormatSummaryData(handlerCache,order)
        res = await self.pushBookKeepOperate.push(summary,requestTime,realRequestTime)
        if res == -1:
            #记账失败发送告警
            try:
                self.pushBookKeepOperate.PushError(summary["payNo"])
            except Exception as e:
                logger.error(f"记账异常处理失败")

    async def PushQueue(self,payNo:str,transacNo:str,transacBody:dict):
        pushDate = FORMAT_TIME("%Y-%m-%d %H:%M:%S")
        title = f"【{AlarmLevel.SERIOUS.value}-{AlarmType.PUSH_TRADE_DATE_FAIL.value}】"
        alarmMsg = f"订单编号:   {payNo}\n交易时间:   {transacBody['programStartTime']}\n上传结果:   失败\n上传时间:   {pushDate}"
        alarmMsgManager = AlarmMsgManager(title=title,msg=alarmMsg)
        self.pushQueueOperate.failAlarmManger = alarmMsgManager
        self.pushQueueOperate.pushUrl = Api.PUSH_IN_QUEUE.value
        res = await self.pushQueueOperate.push(payNo,transacNo,transacBody)
        if res == -1:
            try:
                errRes = await self.pushQueueOperate.PushError(payNo,transacNo,transacBody)
            except Exception as e:
                self.pushQueueOperate.SandPushFailAlarm()
                self.pushQueueOperate.pushStatus = 2
                logger.error(f"第二次入队异常:{e}")
                return
            if errRes == -1:
                    self.pushQueueOperate.SandPushFailAlarm()
                    self.pushQueueOperate.pushStatus = 2
                    logger.error(f"第二次入队失败,已发送告警")
            self.pushQueueOperate.pushStatus = 1
        else:
            self.pushQueueOperate.pushStatus = 0
