import re
import time

from cacheout import LFUCache
from enum import Enum
from typing import Any
from retrying_async import retry
from apscheduler.schedulers.asyncio import AsyncIOScheduler
from tornado.tcpclient import TCPClient
from tornado.iostream import IOStream,StreamClosedError

from lib.struct import PayResSignal,Address
from lib.frame import RsuFrame
from lib.handler import HADLER_MAP,HandlerErrCode
from lib.push import PushOperateManager,Api as PushApi,PushBookKeepOperate
from src.init import TaskSelect
from set import LANE,ATTEMPTS,DELAY
from utils import split,FrameFilter
from log import Logger
logger = Logger(__name__)

class Api(Enum):
    API_PUSH_DEV_ONLANE = "/api/v1/dev/platform/park/dev/status/push/"
    API_GET_WORK_KEY = "/api/v1/gateway/platform/public/get/work/key/"

class ResSelect(Enum):
    E00 = "ok"
    E01 = "E01-天线故障"
    E02 = "E02-未装ETC"
    E03 = "E03-ETC拆卸"
    E04 =" E04-ETC未插卡"
    E05 = "E05-ETC未绑定"
    E06 = "E06-ETC失效"
    E07 = "E07-车牌不符"
    E08 = "E08-非陕西卡"
    E09 = "E09-ETC黑名单"
    E10 = "E10-余额不足"
    E11 = "E11-交易失败"
    E12 = "E12-交易超时"
    E13 = "E13-未知"
    E14 = "E14-读卡错误"
    E15 = "E15-云端不通"
    E16 = "E16-云端异常"
    E17 = "E17-超额消费"
    E18 = "E18-取TAC失败"
    E19 = "E19-记账B卡"


@retry(attempts=ATTEMPTS, delay=DELAY)
async def ConnectRsu(rsuAddress:Address)->IOStream:
    '''
    @rsuAddress 天线地址对象包含IP于服务端口
    return 连接句柄stream
    连接RSU天线
    '''
    logger.info("尝试连接到RSU...")
    stream = await TCPClient().connect(rsuAddress.ip, rsuAddress.port, timeout=2)
    # stream.set_close_callback(close_callback)
    logger.info("链接完成...")
    return stream

async def handler(rf:RsuFrame,stream:IOStream,cache:LFUCache,hadnlerCache:LFUCache)->Any:
    HADNLER_FUN = HADLER_MAP.get(rf.cmdType.hex())
    if not HADNLER_FUN:
        logger.error(f"捕获到未定义帧代码:{rf.cmdType.hex()}")
        return HandlerErrCode.UNKNOWN_ERR.value
    handlerRes = await HADNLER_FUN(stream,cache,hadnlerCache,rf)
    return handlerRes


async def start(stream:IOStream,scheduler:AsyncIOScheduler,payResSignal:PayResSignal,cache:LFUCache,handlerCache:LFUCache)->Any:
    '''
    @stream 天线连接句柄
    主程序开始执行
    '''
    while True:
        fr = await stream.read_bytes(1024, partial=True)
        logger.info(re.sub(r"(?<=\w)(?=(?:\w\w)+$)", " ", fr.hex()))
        try:
            frGroup = split(fr) #分割处理粘包
        except Exception as e:
            logger.error(f"分割buf异常:{e}")
            continue
        frGroup = FrameFilter(frGroup) #过滤非法帧,在有交易帧时过滤心跳帧。
        for f in frGroup:
            rf = RsuFrame(f)
            if rf.isErr:
                logger.error(f"帧数据异常:{rf.errMsg}")
                payResSignal.SetPayResSignal(ResSelect.E14.value,LANE)

            cmdType = rf.cmdType.hex()
            try:
                handlerRes = await handler(rf,stream,cache,handlerCache)
            except Exception as e:
                logger.error(f"帧处理时未捕获异常:{e}")
                if cmdType in ["d0","d1","d2"]:
                    scheduler.resume_job(TaskSelect.TASK_HEARTBEAT.value)  # 恢复心跳
                    continue

            if handlerRes == HandlerErrCode.RUN_ERR.value:
                logger.error(f"处理{cmdType}帧时捕获异常:{handlerRes}")
                payResSignal.SetPayResSignal(ResSelect.E13.value,LANE)

            elif handlerRes == HandlerErrCode.EXCESS_ERR.value:
                logger.info("车辆超额消费")
                payResSignal.SetPayResSignal(ResSelect.E02.value,LANE)

            elif handlerRes == HandlerErrCode.NOT_SELECT_OBU.value:
                logger.info("没有搜索到OBU...")
                payResSignal.SetPayResSignal(ResSelect.E02.value,LANE)

            elif handlerRes == HandlerErrCode.OK.value:
                pass

            elif handlerRes == HandlerErrCode.PAY_OK.value:
                # 开始记账
                ORDER = cache.get("order")
                payResSignal.SetPayResSignal(ResSelect.E00.value,LANE) # 设置扣款成功信号
                logger.info(f"订单:{ORDER['payNo']}完成;{ResSelect.E00.value}")
                requestTime = ORDER.get("requestTime",int(time.time()))
                realRequestTime = ORDER["realRequestTime"]
                pushBookKeepOperate = PushBookKeepOperate(PushApi.PUSH_BOOK_KEEP.value)
                pushOperateManager = PushOperateManager(pushBookKeepOperate=pushBookKeepOperate)
                await pushOperateManager.PushBookKeep(handlerCache,ORDER,requestTime,realRequestTime)
            elif handlerRes == HandlerErrCode.STATUS_ERR.value: # 帧状态异常
                payResSignal.SetPayResSignal(ResSelect.E14.value,LANE)

            elif handlerRes == HandlerErrCode.PLATE_NOT_MATCH.value: # 车牌不符
                ORDER = cache.get("order")
                logger.error(f"车牌不符:{ORDER['plate']}")
                payResSignal.SetPayResSignal(ResSelect.E07.value,LANE)

            elif handlerRes == HandlerErrCode.NET_WORK_ERR.value: # 云端网络异常
                logger.error("请求云端异常")
                payResSignal.SetPayResSignal(ResSelect.E15.value,LANE)

            elif handlerRes == HandlerErrCode.PUSH_ERR.value:
                # 执行到这里说明云入队,本地入队都已失败,直接发送告警
                pushBookKeepOperate = PushBookKeepOperate()
                pushBookKeepOperate.PushError(cache.get("order"))
            else:
                logger.error(f"未捕获的异常状态:{handlerRes}")

            if handlerRes != HandlerErrCode.OK.value:
                scheduler.resume_job(TaskSelect.TASK_HEARTBEAT.value)  # 恢复心跳
            continue