import time
import json
from db.database import mysqlClient,mongoClient
from lib.porxy import PlatformProxy

from log import log
logger = log(__name__)

class OrderProcessor(object):
    def __init__(self, api,date):
        self.api = api
        self.date = date
        self.orders = []
        self.order_ids = []


    async def get_orders_to_submit(self):
        """
        从数据库中获取需要提交的订单
        """
        time_ = int(time.time())
        sql = f'''
        SELECT bk.payNo AS payNo,
            bk.transNumber AS transNumber,
            bk.insertTime AS insertTime,
            bk.isUpload AS isUpload,
            bk.isDelete AS isDelete,
            bk.insertDate AS insertDate,
            bk.insertHour AS insertHour,
            bk.devNumber AS devNumber
        FROM book_keep.book_keep AS bk
        LEFT JOIN res_query rq
            ON bk.payNo = rq.payNo
        LEFT JOIN book_keep.trade_revoke tr
            ON bk.payNo = tr.payNo
        LEFT JOIN book_keep.callback_res cr
            ON bk.payNo = cr.pay_no
        WHERE bk.insertDate = %s
            AND bk.isUpload = %s
            AND {time_} - bk.insertTime >= %s
            AND bk.isDelete = %s
            AND rq.res = %s
            AND cr.callback_status = %s
            AND tr.revokeTime is null
        '''
        self.param = (self.date, 0, 20, 0,"ok",1)
        cur = await mysqlClient.execute(sql, self.param)  # 查询当天且没有提交的记账记录,且保存记录要大于20秒留出撤销时间
        formatted_orders = await cur.fetchall()
        return formatted_orders

    def format_orders(self, orders):
        """
        批量处理订单格式
        """
        format_orders = []
        self.orders = [order["payNo"] for order in orders]
        where = {"payNo":{"$in":self.orders}}
        orders_by_mongo = mongoClient.query("book_keep_content","order",where=where)
        for order_info in orders_by_mongo:
            format_orders.append(order_info["content"])  # 取content部分,就是平台需要记账的字段
        return format_orders

    async def submit_orders(self, formatted_orders):
        """
        批量提交订单
        """
        if not formatted_orders:
            return -1
        http_porxy = PlatformProxy()
        try:
            res = await http_porxy.post(self.api, {"book_keep":formatted_orders}, timeOut=3)
        except Exception as e:
            logger.error(f"提交时网关异常:{e}")
            return -1
        return res

    async def update_order_status(self, orders, status):
        """
        更新订单状态为完成
        """
        if not orders:
            return
        sql = '''update book_keep set isUpload = %s where payNo in %s'''
        cur = await mysqlClient.execute(sql, (status,orders))
        logger.info(f"本次更新{cur.rowcount}条记录")


class FailOrderProcessor(OrderProcessor):
    def __init__(self,api,date) -> None:
        super(FailOrderProcessor,self).__init__(api,date)


    def format_orders(self, orders):
        """
        批量处理订单格式
        """
        format_orders = []
        for order in orders:
            try:
                dic = {
                    "pay_no":order["payNo"],
                    "transac_no":order["transNumber"],
                    "json_str":bytes(json.dumps(order['content']), encoding='gbk').hex()
                }
            except Exception as e:
                logger.error(f"json格式转换错误:{e}")
                continue
            format_orders.append(dic)
        return format_orders

    
    async def get_orders_to_submit(self):
        qs = await mysqlClient.query("trans_failed",  ["id", "payNo"], {"insertDate": self.date, "isUpload": 0})  # 查询当天且没有提交的记账记录
        self.orders = [row["payNo"] for row in qs]
        where = {"payNo":{"$in":self.orders}}
        formatted_orders = mongoClient.query("failed_trans_content","order",where=where)
        return formatted_orders

    
    async def submit_orders(self, formatted_orders):
        """
        批量提交订单
        """
        if not formatted_orders:
            return -1
        http_porxy = PlatformProxy()
        try:
            res = await http_porxy.post(self.api, {"trans_list":formatted_orders}, timeOut=3)
        except Exception as e:
            logger.error(f"提交时网关异常:{e}")
            return -1
        return res
    

    async def update_order_status(self, status):
        """
        更新订单状态为完成
        """
        if not self.orders:
            return
        sql = '''update trans_failed set isUpload = %s where payNo in %s '''
        cur = await mysqlClient.execute(sql, (status,self.orders))
        logger.info(f"本次更新{cur.rowcount}条记录")